<?php

namespace App\Console\Commands;

use App\Models\FileEntry;
use Common\Files\Services\ConvertHeicToJpeg;
use Illuminate\Console\Command;

class RegenerateHeicThumbnails extends Command
{
    protected $signature = 'files:regenerate-heic-thumbnails';
    protected $description = 'Regenerate thumbnails for HEIC/HEIF images';

    public function handle(): int
    {
        $this->info('Regenerating thumbnails for HEIC/HEIF images...');

        // Use case-insensitive comparison for extensions
        $entries = FileEntry::where(function ($query) {
                $query->whereRaw('LOWER(extension) = ?', ['heic'])
                      ->orWhereRaw('LOWER(extension) = ?', ['heif']);
            })
            ->where('type', 'image')
            ->where('file_size', '>', 500000)
            ->where(function ($query) {
                $query->whereNull('public')
                      ->orWhere('public', false);
            })
            ->get();

        $this->info("Found {$entries->count()} HEIC/HEIF files to process");

        $progressBar = $this->output->createProgressBar($entries->count());
        $successCount = 0;
        $failCount = 0;

        foreach ($entries as $entry) {
            try {
                $this->createHeicThumbnail($entry);
                $successCount++;
            } catch (\Exception $e) {
                $this->error("Failed to create thumbnail for {$entry->name}: {$e->getMessage()}");
                $failCount++;
            }
            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine(2);
        $this->info("Successfully created {$successCount} thumbnails");
        
        if ($failCount > 0) {
            $this->warn("Failed to create {$failCount} thumbnails");
        }

        return 0;
    }

    private function createHeicThumbnail(FileEntry $entry): void
    {
        $file = $entry->getDisk()->readStream($entry->getStoragePath());
        $converter = new ConvertHeicToJpeg();
        
        $jpegData = $converter->convertAndResize($file, 350, 250);
        
        $entry
            ->getDisk()
            ->put("{$entry->file_name}/thumbnail.jpg", $jpegData, [
                'mimetype' => 'image/jpeg',
                'visibility' => 'public',
            ]);
        
        $entry->fill(['thumbnail' => true])->save();
    }
}
