<?php

namespace App\Console\Commands;

use App\Models\FileEntry;
use Illuminate\Console\Command;

class FixFileEntryTypes extends Command
{
    protected $signature = 'files:fix-types';
    protected $description = 'Fix MIME types and file types for HEIC, HEIF, and MOV files';

    public function handle(): int
    {
        $this->info('Fixing file entry types for HEIC, HEIF, and MOV files...');

        // Fix HEIC files
        $heicCount = $this->fixFileType('heic', 'image/heic', 'image');
        $this->info("Fixed {$heicCount} HEIC files");

        // Fix HEIF files
        $heifCount = $this->fixFileType('heif', 'image/heif', 'image');
        $this->info("Fixed {$heifCount} HEIF files");

        // Fix MOV files
        $movCount = $this->fixFileType('mov', 'video/quicktime', 'video');
        $this->info("Fixed {$movCount} MOV files");

        $totalCount = $heicCount + $heifCount + $movCount;
        $this->info("Total files fixed: {$totalCount}");

        return 0;
    }

    private function fixFileType(string $extension, string $correctMime, string $correctType): int
    {
        // Use case-insensitive comparison for extension
        $entries = FileEntry::whereRaw('LOWER(extension) = ?', [strtolower($extension)])
            ->where(function ($query) use ($correctMime, $correctType) {
                $query->where('mime', '!=', $correctMime)
                    ->orWhere('type', '!=', $correctType);
            })
            ->get();

        $count = 0;
        foreach ($entries as $entry) {
            $updated = false;
            
            if ($entry->mime !== $correctMime) {
                $entry->mime = $correctMime;
                $updated = true;
            }
            
            if ($entry->type !== $correctType) {
                $entry->type = $correctType;
                $updated = true;
            }
            
            if ($updated) {
                $entry->save();
                $count++;
            }
        }

        return $count;
    }
}
